function fHeader = timeIntegration(modelParam,integParam,rsParam,header)
%function fHeader = timeIntegration(modelParam,integParam,rsParam,header)
%   This function proposes an implementation of the event-driven procedure detailed in 
%      Depouhon A., Detournay E. and Denoël V., "Event-driven integration of linear structural 
%      dynamics models under unilateral elastic constraints", CMAME, 2014
%
%   It is to be called by bouncingBar.m, newtonCradle.m or quakeSimulation.m. See the paper appendix 
%   and  these files for an appropriate definition of the input variables.
%
%   Written by A. Depouhon - 12/4/2013
%   Contact: alexandre.depouhon@ulg.ac.be
    
    tic
    
    % Assign model data to local variables
    K       = modelParam.K;
    C       = modelParam.C;
    M       = modelParam.M;
    fHdle   = modelParam.fHdle;
    W       = modelParam.W;
    u0      = modelParam.u0;
    v0      = modelParam.v0;
    g0      = modelParam.g0;
    k_con   = modelParam.k_con;
    
    % Assign contact data - initial gap & contact stiffness can be provided
	% as a vector or as a scalar (it is then replicated for each interface)
    m       = size(K,1);
    n       = size(W,2);
    if ~(length(g0) == n)
        g0      = g0*ones(n,1);
    else
        g0      = g0(:);
    end
    if ~(length(k_con) == n)
        k_con   = full(k_con)*ones(n,1);
    else
        k_con   = k_con(:);
    end
    
    % Define integration scheme parameters
    param   = integParam(1:2);
    hCp     = integParam(3);
    hRed    = integParam(4);
    nRedH   = integParam(5);
    hOut    = integParam(6);
    tf      = integParam(7);
    maxIncr = round(tf/hCp*100);
       
    % Define root-solving parameters
    gTol    = rsParam(1);
    maxIter = rsParam(2);
    tTol    = rsParam(3);
    
    % Initialize system
        % a) Compute constraint status
    [g,dg]  = gaps(W,g0,u0,v0);
    iC      = getActiveConstraints(g);
        % b) Form contact stiffness matrix
    [KCON,fCON] ...
            = formContactData(iC,k_con,W,g0,M,param(1));
        % c) Compute contact force
    fCtc    = zeros(n,1);
    ind     = iC == 1;
    fCtc(ind) ...
            = -k_con(ind).*g(ind);
        % d) Initialize computational vectors
    if param(1) == 2 % Generalized-A > initial acceleration is required
        a0      = M\(fHdle(0)+fCON-(K+KCON)*u0-C*v0);
        old     = [u0 ; v0 ; a0];
    else
        old     = [u0 ; v0];
    end   
    tOld    = 0;
    nIcrRed = 2*nRedH;
    gOld    = g;
    dgOld   = dg;
    
    % Open storage file
    fHeader = strrep(strtrim(header),' ','_');
    fidLog  = fopen([fHeader '.log'],'w');
    fidTime = fopen([fHeader '_time.his'],'W');
    fidDisp = fopen([fHeader '_displacements.his'],'W');
    fidVelo = fopen([fHeader '_velocities.his'],'W');
    fidFlag = fopen([fHeader '_flags.his'],'W');
    fidGap  = fopen([fHeader '_gaps.his'],'W');
    fidNrg  = fopen([fHeader '_nrg.his'],'W');
    fidCtc  = fopen([fHeader '_ctcForces.his'],'W');
    
    fprintf(fidLog,['      *** ' header ' ***\n']);
    fwrite(fidTime,0,'double');
    fwrite(fidDisp,u0,'double');
    fwrite(fidVelo,v0,'double');
    fwrite(fidGap,gOld,'double');
    fwrite(fidFlag,iC,'double');
    fwrite(fidNrg,0.5*(u0'*K*u0+v0'*M*v0),'double');
    fwrite(fidCtc,fCtc,'double');
   
    fprintf(['      *** ' header ' ***\n']);
    prcCpl  = 0;
    
    % Initialize output points
    tOut    = 0;
    
    % Factorize iteration matrices
    h       = hCp;
    [H0,H1] = formIterationMatrices(m,K+KCON,C,M,h,param);
    [L0,U0,P0,Q0] ...
            = lu(H0);
    
    % Loop over time
    incr    = 2;
    
    while true
%         if incr == 1177
%             keyboard
%         end
        % Increment time
        tNew    = tOld+h;
        
        % Form load vector
        ell     = formIterationLoadVector(m,tOld,tNew,fHdle,fCON,param);
        
        % Update state
        if h == hCp % Equivalent to H0 being factorized
            tmp     = L0\(P0*(H1*old+ell));
            new     = Q0*(U0\tmp);
        else
            new     = H0\(H1*old+ell);
        end
        uNew    = new(1:m);
        vNew    = new(1+m:2*m);
        
        % Compute constraints
        [gNew,dgNew] ...
                = gaps(W,g0,uNew,vNew);
              
        % Detect constraint status changes
        [iEvt,hEvt,dfEvt] ...
                = detectEvents(h,gOld,gNew,dgOld,dgNew);

        % Localize constraints
        if ~isempty(iEvt)
            % Localize earliest event
            [iEvt,hNext,tNew,new,gNew,dgNew] ...
                    = localizeEvents(m,K,KCON,C,M,fHdle,fCON,W,g0,param,tOld,old,gOld, ...
                      dgOld,tNew,gNew,dgNew,iEvt,hEvt,dfEvt,gTol,maxIter,fidLog,tTol,hRed);
            uNew    = new(1:m);
            vNew    = new(1+m:2*m);
            
            % Indicate point has been accepted prior to event occurrence
            hFlag   = 1;
        else
            hFlag   = 0;
        end
        
        % Compute contact force
        fCtc    = zeros(n,1);
        ind     = iC == 1;
        fCtc(ind) ...
                = -k_con(ind).*gNew(ind);
                
        % Update constraint flags, if required
        if ~isempty(iEvt)
            for ii = iEvt'
                if iC(ii) == 1
                    iC(ii)  = 0;
                else
                    iC(ii)  = 1;
                end
            end             
            
            % Update contact data
            [KCON,fCON] ...
                    = formContactData(iC,k_con,W,g0,M,param(1));
            
            % Form iteration matrices
            nIcrRed = 1;
            h       = hRed;
            [H0,H1] = formIterationMatrices(m,K+KCON,C,M,h,param);
        else
            if hFlag
                h       = hNext;
                [H0,H1] = formIterationMatrices(m,K+KCON,C,M,h,param);
                nIcrRed = 1;
            else
                nIcrRed = nIcrRed+1;
            end
        end
                        
        % Use large timestep if already in free flight for nRedH increments
        if nIcrRed > nRedH && h < hCp 
            h       = hCp;
            [H0,H1] = formIterationMatrices(m,K+KCON,C,M,h,param);
            [L0,U0,P0,Q0] ...
                    = lu(H0);
        end
        
        % Store data when required (event localized or time output requested)
        if tNew-tOut >= hOut || ~isempty(iEvt)
            fwrite(fidTime,tNew,'double');
            fwrite(fidDisp,uNew,'double');
            fwrite(fidVelo,vNew,'double');
            fwrite(fidGap,gNew,'double');
            fwrite(fidFlag,iC,'double');
            fwrite(fidNrg,0.5*(uNew'*K*uNew+vNew'*M*vNew),'double');
            fwrite(fidCtc,fCtc,'double');
            tOut    = tNew;
        end
        
        % Print computation progression
        if floor(tNew/tf*100) > prcCpl    
            prcCpl  = floor(tNew/tf*100);
            fprintf(' > Computation completed at %2i %%\n',prcCpl);
        end
               
        % Update old data
        tOld    = tNew;
        old     = new;
        gOld    = gNew;
        dgOld   = dgNew;
        incr    = incr+1;
            
        % Verify inLoop conditions
        if tNew > tf || incr > maxIncr
            % Write output at final step
            fwrite(fidTime,tNew,'double');
            fwrite(fidDisp,uNew,'double');
            fwrite(fidVelo,vNew,'double');
            fwrite(fidGap,gNew,'double');
            fwrite(fidFlag,iC,'double');
            fwrite(fidNrg,0.5*(uNew'*K*uNew+vNew'*M*vNew),'double');
            fwrite(fidCtc,fCtc,'double');
            if incr > maxIncr
                warning('Maximum number of increments has been reached.')
            end
            break
        end
    end
  
    cpTime  = toc;
    
    fprintf('\n > TOTAL NUMBER OF INCREMENTS: %i\n',incr-1);
    fprintf(fidLog,' > TOTAL NUMBER OF INCREMENTS: %i\n',incr-1);
    fprintf(' > TOTAL COMPUTATION TIME: %10.2f s\n',cpTime);
    fprintf(fidLog,' > TOTAL COMPUTATION TIME: %10.2f s\n',cpTime);
    
    fclose('all');
    
end

%%%
%%% AUXILIARY FUNCTIONS
%%%

function iC = getActiveConstraints(g)
    
    iC      = zeros(size(g));
    iC(g < 0) ...
            = 1;

end

function [g,dg] = gaps(W,g0,u,v)

    g       = W'*u+g0;
    dg      = W'*v;
    
end

function [iEvt,hNext,tTmp,newTmp,gTmp,dgTmp] = localizeEvents(m,K,KCON,C, ...
    M,fHdle,fCON,W,g0,param,tOld,old,gOld,dgOld,tNew,gNew,dgNew,iEvt,hEvt, ...
    dfEvt,gTol,maxIter,fidLog,tTol,hRed)
            iter    = 1;
            iEvtIni = iEvt;
            hNext   = [];
            if sum(iEvt == iEvt(1)) > 1
                hMax    = 0.5*sum(hEvt(iEvt == iEvt(1)));
            else
                hMax    = (tNew-tOld);
            end
            hTmp    = min([hMax hEvt(1)+gTol/abs(dfEvt(1))]);
            
            % Define storage array [cpTime ; f ; df]
            cstrAry = nan(1+2*length(gOld),maxIter+1);
            cstrAry(:,1) ...
                    = [tOld ; gOld ; dgOld];
            cstrAry(:,2) ...
                    = [tNew ; gNew ; dgNew];
            colEvt  = 2;

            while true
                % Form iteration matrices and load at event time
                tTmp    = tOld+hTmp;
                [H0,H1] = formIterationMatrices(m,K+KCON,C,M,tTmp-tOld,param);
                ell     = formIterationLoadVector(m,tOld,tTmp,fHdle,fCON,param);

                % Update temporary state
                newTmp  = H0\(H1*old+ell);
                uTmp    = newTmp(1:m);
                vTmp    = newTmp(1+m:2*m);
                                
                % Compute constraints
                [gTmp,dgTmp] ...
                        = gaps(W,g0,uTmp,vTmp);
                
                % Detect constraint status changes in left subinterval
                [iTmp,hTmp,dgEvtTmp] ...
                        = detectEvents(tTmp-tOld,gOld,gTmp,dgOld,dgTmp);

                if isempty(iTmp) 
                    % No significant event takes place in the left subinterval
                    % i.e, no constraint crossing has occurred. An additional 
                    % increment is required. Define hNext unless constraint is exactly zero.
                    iEvt    = find(gTmp == 0);
                    if isempty(iEvt)
                        [iTmp,hTmp,dgEvtTmp] ...
                            = detectEvents(tNew-tTmp,gTmp,gNew,dgTmp,dgNew);
                        if isempty(iTmp) 
                            % No event in right subinterval > numerical grazing
                            % Return to hRed
                            hNext   = hRed;
                        else
                            hNext   = max([sqrt(eps*tTol) min([tNew-tTmp hTmp(1)+0.5*gTol/abs(dgEvtTmp(1))])]);
                        end
                    end
                    break

                else
                    % At least one event takes place in the left subinterval.
                    % Zero crossing is achieved for at least one event.
                    
                    % Get event indices that are within accumulation
                    % distance aTol of the computed point.
                    indTmp  = abs((tTmp-tOld)-hTmp)  < min([gTol/abs(dgTmp(iTmp(1))) tTol]);
                    
                    % If any
                    if sum(indTmp) > 0
                        % Identify the ones that are within the event tolerance gTol
                        iEvt    = iTmp(abs(gTmp(iTmp(indTmp))+gTol*sign(gOld(iTmp(indTmp)))) <= gTol);
                        if ~isempty(iEvt) 
                            % Events have been localized
                            break
                        end
                    end
                    if sum(iTmp == iTmp(1)) > 1
                        hMax    = 0.5*sum(hTmp(iTmp == iTmp(1)));
                    else
                        hMax    = (tTmp-tOld);
                    end
                    scFctr  = ((maxIter-iter)/maxIter)^(100*maxIter/(maxIter-iter));
                    hTmp    = max([sqrt(eps*tTol) min([hMax hTmp(1)+scFctr*gTol/abs(dgEvtTmp(1))])]);
                    colInd  = colEvt:iter+1;
                    cstrAry(:,colInd+1) ...
                            = cstrAry(:,colInd);
                    cstrAry(:,colEvt) ...
                            = [tTmp ; gTmp ; dgTmp];

                    iEvt    = iTmp;
                end

                iter    = iter+1;
                if iter > maxIter
                    fclose('all');
                    error('EVENT-LOCALIZATION DID NOT CONVERGE')
                end
            end
            
            
            fprintf(fidLog,'>> tEvt = %16.15e -- iter = %3i',tTmp,iter);
            if ~isempty(iEvt)
                for ii = 1:length(iEvt)
                   fprintf(fidLog,' -- iEvt = %3i -- gRes = %4.3e',iEvt(ii),abs(gTmp(iEvt(ii))));
                end
                fprintf(fidLog,'\n');
            else
                for ii = 1:length(iEvtIni)
                   fprintf(fidLog,' -- iEvt = %3i -- belongs to right sub-interval',iEvtIni);
                end
                fprintf(fidLog,'\n');
            end
            if hNext < 0
                keyboard
            end

end

function [iEvt,hEvt,dfEvt] = detectEvents(h,gOld,gNew,dgOld,dgNew)
    
    persistent degTol nCon iList
    if isempty(degTol)
        % Set degree degeneracy tolerance
        degTol  = 1e-12;
        nCon    = length(gOld);
        iList   = reshape(repmat(1:nCon,3,1),3*nCon,1);
    end

    % Form Hermite polynomial coefficients
    a3      = 2*gOld+h*dgOld-2*gNew+h*dgNew;
    a2      = -3*gOld-2*h*dgOld+3*gNew-h*dgNew;
    a1      = h*dgOld;
    a0      = gOld;
    
    % Initialize vector
    xiEvt   = Inf(3*nCon,1);  
      
    % Loop over event functions to compute roots (event times)
    for mm = 1:nCon
        if abs(a3(mm)) >= degTol
            A       = diag(ones(2,1),-1);
            A(1,:)  = -[a2(mm)/a3(mm) a1(mm)/a3(mm) a0(mm)/a3(mm)];
            SOL     = eig(A);
        else
            % Use barycentric representation
            C0      = [0 0 0 0 gOld(mm) ; 1 0 0 0 h*dgOld(mm) ; 0 0 1 0 gNew(mm) ; 0 0 1 1 h*dgNew(mm) ;-2 -1 2 -1 0];
            C1      = [1 0 0 0 0; 0 1 0 0 0 ; 0 0 1 0 0 ; 0 0 0 1 0 ; 0 0 0 0 0];
            SOL     = eig(C0,C1);
        end
        ind     = SOL > 0 & SOL <= 1 & imag(SOL) == 0;
        lInd    = sum(ind);
        xiEvt((1:lInd)+(mm-1)*3) ...
                = real(SOL(ind));
    end
    
    % Return results
    if all(isinf(xiEvt))
        iEvt    = [];
        hEvt    = [];
        dfEvt   = [];
    else
        ind     = find(~isinf(xiEvt));
        hEvt    = h*xiEvt(ind);
        iEvt    = iList(ind);
        [hEvt,ind] ...
                = sort(hEvt);
        iEvt    = iEvt(ind);
        dfEvt   = zeros(size(iEvt));
        % Compute derivative at zero crossing
        for ii = 1:length(iEvt)
            dfEvt(ii)   = polyval([3*a3(iEvt(ii)) 2*a2(iEvt(ii)) a1(iEvt(ii))],hEvt(ii)/h)/h;
        end
    end
end

function [H0,H1] = formIterationMatrices(m,K,C,M,h,param)
    
    switch param(1)
        case 0 % Trapezoidal scheme
            ident   = speye(m);
            H0      = [0.5*h*K M+0.5*h*C ; ident -0.5*h*ident];
            H1      = [-0.5*h*K M-0.5*h*C ; ident 0.5*h*ident];
        case 1 % 2-stage BoTr
            rho     = param(2);
            beta    = (1-rho)/(1+rho);
            H0      = [C+(0.5+beta/6)*h*K M-(1/12+beta/12)*h^2*K ; M-(1/12+beta/12)*h^2*K -(0.5+beta/6)*h*M-(1/12+beta/12)*h^2*C];
            H1      = [C-(0.5-beta/6)*h*K M-(1/12-beta/12)*h^2*K ; M-(1/12-beta/12)*h^2*K (0.5-beta/6)*h*M-(1/12-beta/12)*h^2*C];
        case 2 % Generalized-A
            null    = sparse(m,m);
            ident   = speye(m);
            rho     = param(2);
            a_m     = (2*rho-1)/(rho+1);
            a_f     = rho/(rho+1);
            beta    = 0.25*(1-a_m+a_f)^2;
            gamma   = 0.5-a_m+a_f;
            H0      = [(1-a_f)*K (1-a_f)*C (1-a_m)*M ; null ident -gamma*h*ident ; ident null -beta*h^2*ident];
            H1      = [-a_f*K -a_f*C -a_m*M ; null ident (1-gamma)*h*ident ; ident h*ident (0.5-beta)*h^2*ident];
    end

end

function ell = formIterationLoadVector(m,tOld,tNew,fHdle,fCON,param)

    h       = tNew-tOld;
    switch param(1)
        case 0 % Trapezoidal scheme
            ell     = [0.5*h*(fHdle(tOld)+fHdle(tNew))+h*fCON ; zeros(m,1)];
        case 1 % 2-state BoTR -- Simpson-Cavalieri quadrature
            rho     = param(2);
            beta    = (1-rho)/(1+rho);
            fOld    = fHdle(tOld)+fCON;
            fMid    = fHdle(0.5*(tOld+tNew))+fCON;
            fNew    = fHdle(tNew)+fCON;
            I1      = h/6*(fOld+fNew+4*fMid);
            I2      = h/6*(fOld*tOld+fNew*tNew+2*(tNew+tOld)*fMid);
            ell     = [I1 ; (0.5*(tOld+tNew)-beta*h/6)*I1-I2];
        case 2 % Generalized-A
            rho     = param(2);
            a_f     = rho/(rho+1);
            tCp     = tNew*(1-a_f)+tOld*a_f;
            ell     = [fHdle(tCp)+fCON ; zeros(2*m,1)];
    end
    
end

function [KCON,fCON] = formContactData(iC,k_con,W,g0,M,schemeID)

    persistent p_ctcDof
    
    % Compute number of degrees of freedom involved in contact interactions
    if isempty(p_ctcDof)
        nCon    = size(W,2);
        p_ctcDof= nan(nCon,2);
        for ii = 1:nCon
            tmp     = find(W(:,ii)); 
            p_ctcDof(ii,1:length(tmp)) ...
                    = tmp; 
        end
    end

    % Compute number of non-zero entries of contact stiffness matrix
    nnz     = 4*sum(iC)-3*sum(any(isnan(p_ctcDof(logical(iC),:)),2));

    % Compute stiffness matrix and contact load vector
    m   = size(W,1);
    KCON    = sparse([],[],[],m,m,nnz);
    fCON    = sparse(m,1);
    for ii = 1:length(iC)
        if iC(ii)
            KCON    = KCON+k_con(ii)*(W(:,ii)*W(:,ii)');
            fCON    = fCON-k_con(ii)*W(:,ii)*g0(ii);
        end
    end
    if schemeID == 3 && any(iC)
        KCON    = M\KCON;
        fCON    = M\fCON;
    end
end
