%Droplet Tracking
%-----------------

%Input parameters
%----------------
% C1,C2 = The droplet centroid coordinates on the first and second frame
% respectively
% D1, D2 = The droplet diameter on the first and second frame respectively
% image = the name of the frame post processed in order to keep a track of
% which droplet has been measured on which frame
% reso = spatial resolution in µm/pixel
% diffdia = the maximum relative difference in diameter accepted
% angle = the maximum angle between the droplet trajectory and the vertical
% [°]
% vmax = the maximum velocity in the spray
% delay = the delay between two frames
% inf = the focus parameter of each droplet 

%Output parameters
%----------------

% Droplet = a structure containing information about the droplet size,
% displacement, velocity, frame, focus degree
% DropletRej = a structure containing the droplet which can't be tracked.
% In this case only the size, the location, the frame and the focus
% parameter is save


%Output parameters
%---------------
function [Droplet,DropletRej]=Tracking(C1,D1,C2,D2,image,reso,diffdia,angle,vmax,delay,inf)

g=0;
rej=0;
Droplet=[];
DropletRej=[];
depmax=vmax*delay/reso; %Maximum displacement allowed
angle=angle*pi/180;     %Maximal angle allowed between the vertical and the droplet direction [rad]
if abs(isempty(C1)-1) && abs(isempty(C2)-1)  
    for j=1:size(C1,1) 
        ind=[];
        
        %Computation of the criteria of displacement
        c1=[C1(j,1)*ones(size(C2,1),1) C1(j,2)*ones(size(C2,1),1)];
        Dep=c1-C2;                         
        dx=Dep(:,1);                             %Distance x
        dy=Dep(:,2);                             %Distance y
        dist=sqrt(dx.^2+dy.^2);                %Distance between the Droplets
        ind1=find(atan(abs(dx./dy))<angle & dy<0 & dist<depmax); %Index of the droplet fullfilling the criteria
        if ~isempty(ind1)
            %computation of the size criteria
            d1=D1(j)*ones(size(ind1,2),1);
            deltadia=abs(d1'-D2(ind1))/d1(1);           %Diameter difference between the Droplets
            ind2=find(deltadia<diffdia);                  
            ind=ind1(ind2);
            
            if length(ind)>1            
                [~, i]=min(deltadia(ind2));
                ind=ind1(i);
            end  
        end
        
        %Structure containing the droplet characteristics

        if isempty(ind)
            rej=rej+1;
            DropletRej(rej).diametre=D1(j);
            DropletRej(rej).centre=C1(j,:);
            DropletRej(rej).image=image;
            DropletRej(rej).inf=inf(j);

        else 
            
            g=g+1;
            Droplet(g).diametre=D1(j);
            Droplet(g).diametreAvg=(deltadia(ind2));
            Droplet(g).volume=((D1(j)/2)^3)*pi*4/3;
            Droplet(g).displacement=dist(ind);
            Droplet(g).centre=[C1(j,:);C2(ind,:)];
            Droplet(g).vit=dist(ind)*reso/delay;
            Droplet(g).image=image;
            Droplet(g).inf=inf(j);

        end
        
    end

end
end